/***************************************************************************
 *
 *  Oxford Semiconductor Proprietary and Confidential Information
 *
 *  Copyright:      Oxford Semiconductor Ltd, 2006
 *
 *  Description:    OS independant HW access definitions.
 *
 ***************************************************************************/

#ifndef INC_HwAccess_h
#define INC_HwAccess_h


#include "OsHwAccess.h"



/**
 *  Initialise a register base.
 * @param A register base descriptor.
 * @param A pointer to a partial HW resource descriptor.
 */
void
OsRegBaseInit(
    OsRegBaseDescriptor * const pRegBaseDescr,
    OsHwResource  const * const pHwResource);


/**
 *  Get the number of bytes occupied by a register base.
 * @param A register base descriptor.
 * @return Register base length.
 */
size_t
OsRegBaseGetLength(
    OsRegBaseDescriptor const * const pRegBaseDescr);


/**
 *  Map the given register base to nonpaged system space
 *  if it is in memory (and not in I/O space).
 * @param A register base descriptor.
 * @return OS_STATUS__OK or OS_STATUS__NO_RESOURCES.
 */
OsStatus
OsRegBaseMap(
    OsRegBaseDescriptor * const pRegBaseDescr);


/**
 *  Unmap the given register base from nonpaged system space (if mapped).
 * @param A register base descriptor.
 */
void
OsRegBaseUnmap(
    OsRegBaseDescriptor * const pRegBaseDescr);


/**
 *  Read an 8-bit wide register.
 * @param A register base descriptor.
 * @param A register offset in bytes.
 * @return Value read from the register.
 */
U08
OsRegisterRead08(
    OsRegBaseDescriptor const * const pRegBaseDescr,
    size_t                            offset);


/**
 *  Read a 16-bit wide register.
 * @param A register base descriptor.
 * @param A register offset in bytes.
 * @return Value read from the register.
 */
U16
OsRegisterRead16(
    OsRegBaseDescriptor const * const pRegBaseDescr,
    size_t                            offset);


/**
 *  Read a 32-bit wide register.
 * @param A register base descriptor.
 * @param A register offset in bytes.
 * @return Value read from the register.
 */
U32
OsRegisterRead32(
    OsRegBaseDescriptor const * const pRegBaseDescr,
    size_t                            offset);


/**
 *  Write to an 8-bit wide register.
 * @param A register base descriptor.
 * @param A register offset in bytes.
 * @param A value to be written.
 */
void
OsRegisterWrite08(
    OsRegBaseDescriptor const * const pRegBaseDescr,
    size_t                            offset,
    U08                               value);


/**
 *  Write to a 16-bit wide register.
 * @param A register base descriptor.
 * @param A register offset in bytes.
 * @param A value to be written.
 */
void
OsRegisterWrite16(
    OsRegBaseDescriptor const * const pRegBaseDescr,
    size_t                            offset,
    U16                               value);


/**
 *  Write to a 32-bit wide register.
 * @param A register base descriptor.
 * @param A register offset in bytes.
 * @param A value to be written.
 */
void
OsRegisterWrite32(
    OsRegBaseDescriptor const * const pRegBaseDescr,
    size_t                            offset,
    U32                               value);



#include "OsHwAccess_inline.h"



#endif // INC_HwAccess_h
