/***************************************************************************
 *
 *  Oxford Semiconductor Proprietary and Confidential Information
 *
 *  Copyright:      Oxford Semiconductor Ltd, 2006
 *
 *  Description:    Definition of the GPIO core functionality.
 *
 ***************************************************************************/

#ifndef INC_GpioCore_h
#define INC_GpioCore_h

#include "Common.h"
#include "GpioCoreDef.h"



/**
 *  Construct a GPIO core object.
 * @param GPIO core parent device handle.
 * @param Callback function that cancels the I/O request.
 * @return A pointer to newly created GPIO core object, or NULL on failure.
 */
GpioCore *
GpioCoreConstruct(
    OsDeviceHandle        hDevice,
    OsIoctlCancelCallback pfIoctlCancel);


/**
 *  Destruct a GPIO core object.
 * @return A pointer to a GPIO core object.
 */
void
GpioCoreDestruct(
    GpioCore * const pThis);


/**
 *  Assign HW resources to a GPIO core object.
 * @param Pointer to a GPIO core object.
 * @param Pointer to a device register base descriptor.
 * @param An associated interrupt handle.
 * @return OsStatus.
 */
OsStatus
GpioCoreAssignHwResources(
    GpioCore            * const pThis,
    OsRegBaseDescriptor * const pRegBaseDescr,
    OsInterruptHandle           hInterrupt);


/**
 *  Release HW resources from a GPIO core object.
 * @param Pointer to a GPIO core object.
 */
void
GpioCoreReleaseHwResources(
    GpioCore * const pThis);


/**
 *  Interrupt service routine.
 * @param Pointer to a GPIO core object.
 * @return TRUE if the function services the interrupt, FALSE otherwise.
 */
BOOLEAN
GpioCoreIsr(
    GpioCore * const pThis);


/**
 *  Deferred interrupt processing.
 * @param Pointer to a GPIO core object.
 */
void
GpioCoreDpcFunc(
    GpioCore * const pThis);


/**
 *  Enable HW interrupts.
 *  The interrupt lock must already be acquired before calling this function.
 * @param Pointer to a GPIO core object.
 */
void
GpioCoreInterruptEnable(
    GpioCore * const pThis);


/**
 *  Disable HW interrupts.
 *  The interrupt lock must already be acquired before calling this function.
 * @param Pointer to a GPIO core object.
 */
void
GpioCoreInterruptDisable(
    GpioCore * const pThis);


/**
 *  Hanlde I/O control requests.
 * @param Pointer to a GPIO core object.
 * @param Handle to an IOCTL request object.
 *        If OS_INVALID_HANDLE, the function will never return OS_STATUS__PENDING.
 * @param I/O control code.
 * @param Pointer to an input buffer.
 * @param Input buffer size.
 * @param Pointer to an output buffer.
 * @param Output buffer size.
 * @param The number of bytes returned in output buffer.
 * @return OsStatus.
 */
OsStatus
GpioCoreIoctlDispatch(
    GpioCore * const    pThis,
    OsIoctlHandle       hIoctl,
    GpioIoControlCode   ioControlCode,
    void const * const  pInput,
    size_t              inputLength,
    void * const        pOutput,
    size_t              outputLength,
    size_t * const      pBytesReturned);


/**
 *  Cancel I/O control requests.
 * @param Pointer to a GPIO core object.
 * @param Handle to an IOCTL request object.
 */
void
GpioCoreIoctlCancel(
    GpioCore * const pThis,
    OsIoctlHandle    hIoctlToCancel);


/**
 *  Handle Dx->D0 transitions.
 *  This function can be called only during Dx->D0 transitions, when
 *  no interrupts are enabled (i.e. no need to obtain interrupt locks).
 * @param Pointer to a GPIO core object.
 * @param Device power state which the device is about to leave.
 */
void
GpioCoreD0Entry(
    GpioCore * const   pThis,
    OsPowerDeviceState oldDx);


/**
 *  Handle D0->Dx transitions.
 *  This function can be called only during D0->Dx transitions, when
 *  no interrupts are enabled (i.e. no need to obtain interrupt locks).
 * @param Pointer to a GPIO core object.
 * @param Device power state which the device is about to enter.
 */
void
GpioCoreD0Exit(
    GpioCore * const   pThis,
    OsPowerDeviceState newDx);


/**
 *  Arm wakeups from Dy+S0.
 * @param Pointer to a GPIO core object.
 */
void
GpioCoreArmWakeFromS0(
    GpioCore * const pThis);


/**
 *  Disarm wakeups from Dy+S0.
 * @param Pointer to a GPIO core object.
 */
void
GpioCoreDisarmWakeFromS0(
    GpioCore * const pThis);


/**
 *  Handle wakeups from Dy+S0.
 * @param Pointer to a GPIO core object.
 */
void
GpioCoreWakeFromS0Triggered(
    GpioCore * const pThis);


/**
 *  Arm wakeups from Dy+Sx.
 * @param Pointer to a GPIO core object.
 */
void
GpioCoreArmWakeFromSx(
    GpioCore * const pThis);


/**
 *  Disarm wakeups from Dy+Sx.
 * @param Pointer to a GPIO core object.
 */
void
GpioCoreDisarmWakeFromSx(
    GpioCore * const pThis);


/**
 *  Handle wakeups from Dy+Sx.
 * @param Pointer to a GPIO core object.
 */
void
GpioCoreWakeFromSxTriggered(
    GpioCore * const pThis);



#endif // INC_GpioCore_h
